/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier MN2WS0210 PCM Driver");
MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE(
	"{{Panasonic, MN2WS0210}, "
	"{Panasonic, MN2WS0210x}}");


#define MN2WS_PCM_DEV_NAME       "mn2ws0210-pcm"
#define MN2WS_PCM_DEV_COUNT      2

#define MN2WS_RES_PCM            0

#define MN2WS_PCM_REG_START      (0xde00c000)
#define MN2WS_PCM_REG_SIZE       (0x4000)

//tentative...
#define MN2WS_PCM_HW_BUF(ch) \
	((ch == 0) ? 0x816bac00 : \
	(ch == 1) ? 0x816bac00 : \
	0)

#define MN2WS_PCM_HW_BUF_SIZE(ch) \
	((ch == 0) ? (PAGE_SIZE) : \
	(ch == 1) ? (PAGE_SIZE) : \
	0)

//3b10, 3b14, 3b18
#define PCM_PCMOCTL_0         (0xde00fb10)
#define PCM_PCMOCTL_1         (0xde00fb14)
#define PCM_IECOCTL           (0xde00fb18)

//300c, 330c
#define PCM_AUDCTL0(ch) \
	((ch == 0) ? 0xde00f00c : \
	(ch == 1) ? 0xde00f30c : \
	0)

//30c0, 33c0
#define PCM_PCMMIX0(ch) \
	((ch == 0) ? 0xde00f0c0 : \
	(ch == 1) ? 0xde00f3c0 : \
	0)
#define PCM_PCMMIX1(ch) \
	((ch == 0) ? 0xde00f0c4 : \
	(ch == 1) ? 0xde00f3c4 : \
	0)
#define PCM_PCMMIX2(ch) \
	((ch == 0) ? 0xde00f0c8 : \
	(ch == 1) ? 0xde00f3c8 : \
	0)

//3164, 3464
#define PCM_MIXCURADDR(ch) \
	((ch == 0) ? 0xde00f164 : \
	(ch == 1) ? 0xde00f464 : \
	0)


static int mn2ws0210_pcm_play_setup(struct mn2ws_pcm_dev *dev);
static int mn2ws0210_pcm_play_start(struct mn2ws_pcm_dev *dev);
static int mn2ws0210_pcm_play_stop(struct mn2ws_pcm_dev *dev);
static unsigned long mn2ws0210_pcm_play_get_hwptr(struct mn2ws_pcm_dev *dev);
static void mn2ws0210_pcm_play_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr);
static int mn2ws0210_pcm_init_module(void);
static void mn2ws0210_pcm_exit_module(void);


static struct resource mn2ws_resources[] = {
	[MN2WS_RES_PCM] = {
		.start = MN2WS_PCM_REG_START, 
		.end   = MN2WS_PCM_REG_START + MN2WS_PCM_REG_SIZE, 
		.flags = IORESOURCE_MEM, 
	}, 
};

struct mn2ws_pcm_desc mn2ws_pcm[] = {
	//ch 0
	{
		.res_regs      = mn2ws_resources, 
		.n_res_regs    = ARRAY_SIZE(mn2ws_resources), 
		
		.play = {
			.enabled       = 1, 
			//tentative: use fixed address
			.phys_addr_buf = MN2WS_PCM_HW_BUF(0), 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE(0), 
			//0 means use default size(= size_buf)
			//24ms(min: 768bytes = 24ms)
			.size_use_fix  = 128 * 6, 
			.init          = NULL, 
			.term          = NULL, 
			.hardware      = mn2ws_pcm_pcmif_hardware_16k_16b_1ch, 
			.setup         = mn2ws0210_pcm_play_setup, 
			.start         = mn2ws0210_pcm_play_start, 
			.stop          = mn2ws0210_pcm_play_stop, 
			.get_hwptr     = mn2ws0210_pcm_play_get_hwptr, 
			.set_devptr    = mn2ws0210_pcm_play_set_devptr, 
			.wait_hwevent  = mn2ws_pcm_pcmif_hrtimeout, 
			.copy          = mn2ws_pcm_play_copy_to_hw, 
			.silence       = mn2ws_pcm_play_silence_to_hw, 
		}, 
	}, 
	
	//ch 1
	{
		.res_regs      = mn2ws_resources, 
		.n_res_regs    = ARRAY_SIZE(mn2ws_resources), 
		
		.play = {
			.enabled       = 1, 
			//tentative: use fixed address
			.phys_addr_buf = MN2WS_PCM_HW_BUF(1), 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE(1), 
			//0 means use default size(= size_buf)
			//24ms(min: 768bytes = 24ms)
			.size_use_fix  = 128 * 6, 
			.init          = NULL, 
			.term          = NULL, 
			.hardware      = mn2ws_pcm_pcmif_hardware_16k_16b_1ch, 
			.setup         = mn2ws0210_pcm_play_setup, 
			.start         = mn2ws0210_pcm_play_start, 
			.stop          = mn2ws0210_pcm_play_stop, 
			.get_hwptr     = mn2ws0210_pcm_play_get_hwptr, 
			.set_devptr    = mn2ws0210_pcm_play_set_devptr, 
			.wait_hwevent  = mn2ws_pcm_pcmif_hrtimeout, 
			.copy          = mn2ws_pcm_play_copy_to_hw, 
			.silence       = mn2ws_pcm_play_silence_to_hw, 
		}, 
	}, 
};


const char *get_pcm_dev_name(void)
{
	return MN2WS_PCM_DEV_NAME;
}

int get_pcm_dev_count(void)
{
	return MN2WS_PCM_DEV_COUNT;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0210_pcm_play_setup(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mrp = &dev->map_regs[MN2WS_RES_PCM];
	
	//tentative: Main:DEC0, Sub:DEC1, IEC:DEC1
	pcm_writel(mrp, 0x0000c021, PCM_PCMOCTL_0);
	pcm_writel(mrp, 0x0000c029, PCM_PCMOCTL_1);
	pcm_writel(mrp, 0x00100009, PCM_IECOCTL);
	
	pcm_writel(mrp, 0x00240000, PCM_AUDCTL0(dev->ch));
	
	pcm_writel(mrp, 
		dev->play.paddr_buf - 0x80000000, PCM_PCMMIX1(dev->ch));
	pcm_writel(mrp, 
		dev->play.size_buf_use, PCM_PCMMIX0(dev->ch));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0210_pcm_play_start(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mrp = &dev->map_regs[MN2WS_RES_PCM];
	
	DPRINTF("start the playback DMA %d.\n", dev->ch);
	
	//start the transfer DMA
	pcm_writel(mrp, 0x00002882, PCM_PCMMIX2(dev->ch));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0210_pcm_play_stop(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mrp = &dev->map_regs[MN2WS_RES_PCM];
	
	DPRINTF("stop the playback DMA %d.\n", dev->ch);
	
	//stop the transfer DMA
	pcm_writel(mrp, 0x00002880, PCM_PCMMIX2(dev->ch));
	
	return 0;
}

/**
 * PCM  HW ɤ߼äƤХåեˤĤơ
 * ߤ HW ɤ߼֤롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @return HW θߤɤ߼
 */
static unsigned long mn2ws0210_pcm_play_get_hwptr(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mrp = &dev->map_regs[MN2WS_RES_PCM];
	
	return pcm_readl(mrp, PCM_MIXCURADDR(dev->ch)) - 0x80000000;
}

/**
 * PCM  HW ɤ߼äƤХåեˤĤơ
 * ߤΥǥХν񤭹֤߰ꤹ롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @param ptr ǥХθߤν񤭹߰
 */
static void mn2ws0210_pcm_play_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr)
{
	//񤭹֤߰ΤפΤᲿ⤷ʤ
}

static int __init mn2ws0210_pcm_init_module(void)
{
	return mn2ws_pcm_init_module(mn2ws_pcm);
}

static void __exit mn2ws0210_pcm_exit_module(void)
{
	mn2ws_pcm_exit_module(mn2ws_pcm);
}

module_init(mn2ws0210_pcm_init_module);
module_exit(mn2ws0210_pcm_exit_module);
